function [L] = RABT_Setpoints(Pic,N,scrsize,W)
% RABT_Setpoints
% 
% Author: Koen Hilgersom (2011)
% 01/07/2011
%
% Function file is run by the script 'Rising Air Bubble Technique_vX_X.m'
%
% Function runs interactive screen for setting the locations of the
% surfacing bubbles and the intersections of the nozzle line with the
% banks.
%
% Input
% Pic        -    Original picture
% N          -    Number of inputs
% scrsize    -    User screensize
% W          -    Approximate width of the river
%
% Output
% L          -    Struct with marked locations in the original picture
%
%---------------------------------------------------------------------------  
%      Copyright (C) 2011 Technische Universiteit Delft, 
%          Koen Hilgersom
%          K.P.Hilgersom@tudelft.nl (correspondence)
% 
%---------------------------------------------------------------------------  

% create figure with zoomed image
big = figure('Name','Zoomed Image','NumberTitle','off','IntegerHandle','off',...
    'Toolbar','none','Menubar','none'); clf
hIm=imshow(Pic);
hSP = imscrollpanel(big,hIm);
api = iptgetapi(hSP);
if W>10
    api.setMagnification(4);
elseif W<10 && W>7
    api.setMagnification(3);
else
    api.setMagnification(2);
end
set(gcf,'Position',[scrsize(1) scrsize(4)/2+scrsize(2)+30 scrsize(3) scrsize(4)/2-10]);
hold on

% create figure with total image
sma = figure('Name','Total Image','NumberTitle','off','IntegerHandle','off',...
    'Toolbar','none','Menubar','none'); clf
imoverviewpanel(sma,hIm);
set(gcf,'Position',[scrsize(1) scrsize(2)+10 scrsize(3) scrsize(4)/2-10]);
hold on

% input of bubble locations (with green dots)
disp('Please set the locations of the surfacing bubbles by dragging and double-clicking the green dots');
Loc=ones(N+2,2);
for n=1:N
    Loc(n,:) = putPoints(big,sma,'g');
end
if max(Loc(:,1))-min(Loc(:,1))<max(Loc(:,2))-min(Loc(:,2))
    X=2;
else
    X=1;
end
L.bubbleOrig(1:N,:)=sortrows(Loc(1:N,:),X);

% input of side points of the nozzle line (with red dots)
disp('Please set the locations of the nozzle line intersections with the channel banks with the red dots');
for n=N+1:N+2
    Loc(n,:) = putPoints(big,sma,'r');
end
L.SidePOrig(1:2,:)=flipud(sortrows(Loc(N+1:N+2,:),X));
end

function Loc = putPoints(big,sma,C)
    h = impoint(gca,100,100);
    constrainMarker(h,sma,C)
    figure(big)
    p = impoint(gca,50,50);
    constrainMarker(p,big,C)
    apiSma = iptgetapi(h);
    apiBig = iptgetapi(p);
    initial_position = apiSma.getPosition();
    updateLoc(initial_position,p,sma)
    apiBig.addNewPositionCallback(@(p) updateLoc(p,h,sma));
    apiSma.addNewPositionCallback(@(h) updateLoc(h,p,big));
    setString(p,'Double-click to confirm location');
    Loc=wait(p);
    figure(big); scatter(Loc(1),Loc(2),'FaceColor',C,'SizeData',100);
    figure(sma); scatter(Loc(1),Loc(2),'FaceColor',C,'SizeData',10);
    delete(h);delete(p);
end

function constrainMarker(h,fig,C)
    setColor(h,C);
    figure(fig)
    addNewPositionCallback(h,@(h) title(sprintf('(%1.0f,%1.0f)',h(1),h(2))));
    fcn = makeConstrainToRectFcn('impoint',get(gca,'XLim'),get(gca,'YLim'));
    setPositionConstraintFcn(h,fcn);
end

function updateLoc(h,p,a)
    figure(a)
    setPosition(p,h)
end